/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#include "systeminfo.h"
#include "log.h"

#include <QDir>

using namespace osdev::components;

// The only instance of the singleton system info.
std::unique_ptr<SystemInfo> SystemInfo::s_pInstance(nullptr);

// The configuration directory.
static const QString qsConfigDir = "config/";
static const QString qsResourceDir = "resource/";

static const QString qsDynaBuilderSystemPluginConfig;
static const QString qsDynaBuilderUiPluginConfig;
static const QString qsDynaBuilderBackendPluginConfig;


SystemInfo::SystemInfo()
    : m_qsApplicationPath()
    , m_qsExecutablePath(QDir::currentPath())
{
}

SystemInfo::~SystemInfo()
{
}

SystemInfo& SystemInfo::Instance()
{
    // Is it the first call?
    if (nullptr == s_pInstance)
    {
        // Create sole instance
        s_pInstance.reset(new SystemInfo());
    }

    // Return the address of sole instance
    return *s_pInstance.get();
}

void SystemInfo::destroyInstance()
{
    s_pInstance.reset();
}

void SystemInfo::setApplicationPath(const QString& qsName)
{
    QDir dir( qsName );

    m_qsApplicationPath = dir.absolutePath();

    LogInfo("SystemInfo::setApplicationPath",
            "ApplicationPath before :: " + m_qsApplicationPath);

    // Remove appl name.
    int lSlashPosition = m_qsApplicationPath.lastIndexOf ( '/' );

    // And platform path.
    // lSlashPosition = m_qsApplicationPath.lastIndexOf ( '/', lSlashPosition - 1);
    m_qsApplicationPath = m_qsApplicationPath.left( lSlashPosition + 1 );

    QString strBin = "bin";

    m_qsApplicationPath = m_qsApplicationPath.left( m_qsApplicationPath.lastIndexOf( strBin ) );

    LogInfo("SystemInfo::setApplicationPath",
            "ApplicationPath after :: " + m_qsApplicationPath);

    /// @todo The application can be started via the PATH variable. This path
    ///       should be prepended to the path.
}

QString SystemInfo::getApplicationPath() const
{
    return m_qsApplicationPath;
}

QString SystemInfo::getExecutablePath() const
{
    return m_qsExecutablePath;
}

QString SystemInfo::getResourcePath() const
{
    return getApplicationPath() + qsResourceDir;
}

QString SystemInfo::getConfigurationPath() const
{
    return getApplicationPath() + qsConfigDir;
}

QString SystemInfo::getConfigFile(ConfigFile fileType) const
{
    switch(fileType)
    {
    case ConfigFile::System:
        return getApplicationPath() + qsConfigDir + qsDynaBuilderSystemPluginConfig;
    case ConfigFile::UI:
        return getApplicationPath() + qsConfigDir + qsDynaBuilderUiPluginConfig;
    case ConfigFile::Backend:
        return getApplicationPath() + qsConfigDir + qsDynaBuilderBackendPluginConfig;
    }
    Q_ASSERT(0 && "Unknown configuration file type");
    return QString();
}
