/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/
#ifndef OSDEV_COMPONENTS_PLUGIN_H
#define OSDEV_COMPONENTS_PLUGIN_H

#include <QString>
#include <QUuid>

class QObject;

namespace osdev  {
namespace components {

/*!
 * \brief The Plugin class is a structure representing a single plugin.
 *        It is used internally by the Pluginmanager and holds all information
 *        needed.
 *//*
 *   _______________________________________
 *  / Heuristics are bug ridden by          \
 *  | definition. If they didn't have bugs, |
 *  \ then they'd be algorithms.            /
 *   ---------------------------------------
 *     \
 *      \
 *          .--.
 *         |o_o |
 *         |:_/ |
 *        //   \ \
 *       (|     | )
 *      /'\_   _/`\
 *      \___)=(___/
 *
 */

class Plugin
{
public:
    //!
    /**
     * @brief Constructor
     * @param pluginName Name of the plugin
     * @param pluginPath Path to the plugin
     * @param pluginConfig Configuration file for the plugin
     * @param pluginUuid UUID for the plugin
     * @param ptr_plugin Plugin instance
     */
    Plugin(const QString& pluginName = QString(),
           const QString& pluginFile = QString(),
           const QString& pluginPath = QString(),
           const QString& pluginConfig = QString(),
           const QUuid& pluginUuid = QUuid(),
           QObject* ptr_plugin = nullptr );

    /**
     * @brief Copy-constructor
     * @param other Plugin to copy from
     */
    Plugin(const Plugin& other);

    /**
     * @brief Assignment operator
     * @param other Plugin to copy from
     * @return Reference to this
     */
    Plugin& operator=(const Plugin& other);

    /*** Getters and Setters ***/
    //! @returns Name of the plugin ( like : libplugin_myplugin ).
    QString name() const { return m_pluginName; }

    //! @returns Path to the plugin.
    QString path() const { return m_pluginPath; }

    //! @returns the filename of the plugin.
    QString fileName() const { return m_pluginFile; }

    //! @returns Full pathname to the plugin specific configurationfile.
    QString config() const { return m_pluginConfigurationFile; }

    //! @returns Unique Identifier of the plugin.
    QUuid uuid() const { return m_pluginUuid; }

    //! @returns Pointer to the plugin-instance as a QObject.
    QObject* plugin() const { return m_plugin; }

    //! Sets the name of the plugin. Make sure to omit the extension.
    //! @param pluginName Name of the plugin
    void setName( const QString& pluginName ) { m_pluginName = pluginName; }

    //! Sets the filename of the plugin. Make sure we omit the extension.
    //! @param pluginFile Filename of the plugin.
    void setFileName( const QString &fileName ) { m_pluginFile = fileName; }

    //! Sets the path to the plugin.
    //! @param pluginPath Path to the plugin
    //!
    //! PluginManager uses only one pluginpath, but it was added for future use.
    void setPath( const QString& pluginPath ) { m_pluginPath = pluginPath; }

    //! Sets the full pathname and name of the plugin specific configuration file
    //! @param pluginConfig Configuration file for the plugin
    void setConfig( const QString& pluginConfig ) { m_pluginConfigurationFile = pluginConfig; }

    /*! Sets the Unique Identifier of the specific plugin.
     *
     * If the plugin is not yet loaded, its value is
     * {00000000-0000-0000-0000-000000000000} ( or isNull is true ). If a plugin
     * is loaded, its value is changed with the value read from the pluginID.
     *
     * @param pluginUuid UUID for the plugin
     */
    void setUuid( const QUuid& pluginUuid ) { m_pluginUuid = pluginUuid; }

    //! Sets the pointer to the plugin-instance.
    //! @param ptr_plugin Instance pointer
    void setPlugin( QObject* ptr_plugin ) { m_plugin = ptr_plugin; }

    QString asString();

private:
    QString  m_pluginName;              ///< Name of the plugin
    QString  m_pluginFile;              ///< Filename of the plugin
    QString  m_pluginPath;              ///< Full path to the plugin
    QString  m_pluginConfigurationFile; ///< Full path to the configuration file
    QUuid    m_pluginUuid;              ///< Plugin UUID
    QObject* m_plugin;                  ///< Plugin implementation
};

} // End namespace components
} // End namespace osdev

#endif // OSDEV_COMPONENTS_PLUGIN_H
