/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

#ifndef OSDEV_COMPONENTS_DBRELATION_H
#define OSDEV_COMPONENTS_DBRELATION_H

#include <QObject>
#include <QString>

namespace osdev  {
namespace components {

/*!
 * \brief   The DbRelation class is a container class holding the
 *          information regarding the relation between two tables.
 */

class DbRelation : public QObject
{
    Q_OBJECT

public:
    DbRelation( QObject *parent = nullptr );

    /*!
     * \brief   Get the name of the database scheme
     * \return  The schemaname as QString. isEmpty() is true if none was set.
     */
    const QString& schemaName() const { return m_schemaName; }

    /*!
     * \brief   Set the name of the database scheme
     * \param   _schemaName - The name of the schema.
     */
    void setSchemaName( const QString &_schemaName ) { m_schemaName = _schemaName; }

    /*!
     * \brief   Get the name of the table this relation describes.
     * \return  The name of the table this relation describes.
     */
    const QString& tableName() const { return m_tableName; }

    /*!
     * \brief   Set the name of the table this relation describes.
     * \param   _tableName - The name of the table this relation describes.
     */
    void setTableName( const QString &_tableName ) { m_tableName = _tableName; }

    /*!
     * \brief   Get the name of the constraint this relation describes.
     * \return  the name of the constraint this relation describes.
     */
    const QString& constraintName() const { return m_constraintName; }

    /*!
     * \brief   Set the name of the constraint this relation describes.
     * \param   _constraintName - The name of the constraint this relation describes.
     */
    void setConstraintName( const QString &_constraintName ) { m_constraintName = _constraintName; }

    /*!
     * \brief   Get the name of the related table this relation describes.
     * \return  the name of the related table this relation describes.
     */
    const QString& foreignTable() const { return m_foreignTable; }

    /*!
     * \brief   Set the name of the related table this relation describes.
     * \param   _foreignTable - the name of the related table this relation describes.
     */
    void setForeignTable( const QString &_foreignTable ) { m_foreignTable = _foreignTable; }

    /*!
     * \brief   The name of the field on the maintable this relation describes.
     * \return  The name of the field on the maintable this relation describes.
     */
    const QString& indexColumn() const { return m_indexColumn; }

    /*!
     * \brief   Set the name of the field on the maintable this relation describes.
     * \param   _indexColumn - The name of the field on the maintable this relation describes.
     */
    void setIndexColumn( const QString &_indexColumn ) { m_indexColumn = _indexColumn; }

    /*!
     * \brief   Gets the name of the primary key field of the related table.
     * \return  The name of the primary key field of the related table.
     */
    const QString& foreignPrimKey() const { return m_foreignPrimKey; }

    /*!
     * \brief Sets the name of the primary key field of the related table.
     * \param _foreignPrimKey - The name of the primary key field of the related table.
     */
    void setForeignPrimKey( const QString &_foreignPrimKey ) { m_foreignPrimKey = _foreignPrimKey; }

    /*!
     * \brief   Return the relation structure as a QString for Debugging purposes
     * \return  The datastructure in QString representation.
     */
    QString asString() const { return QString( "SchemaName : %1 \nTableName : %2 \n ConstraintName : %3 \n ForeignTable : %4 \n IndexColumn : %5 \n Foreign Primary Key : %6")
                .arg( m_schemaName, m_tableName, m_constraintName, m_foreignTable, m_indexColumn, m_foreignPrimKey ); }

private:
    QString     m_schemaName;       ///< The name of the database scheme
    QString     m_tableName;        ///< The name of the target-table
    QString     m_constraintName;   ///< The name of the constraint as known to the database
    QString     m_foreignTable;     ///< The name of the table the target has a relation with
    QString     m_indexColumn;      ///< The name of the column of the target table that holds the relation.
    QString     m_foreignPrimKey;   ///< The name of the primary key-field of the related table.
};

}       /* End namespace components     */
}       /* End namespace osdev          */

#endif  /* OSDEV_COMPONENTS_DBRELATION_H */
