/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/


#ifndef OSDEV_COMPONENTS_DBCONNECTIONWATCHDOG_H
#define OSDEV_COMPONENTS_DBCONNECTIONWATCHDOG_H

#include <QObject>
#include <QTimer>
#include <QPointer>

namespace osdev {
namespace components {
/*!
 * \brief   The DbConnectionWatchDog class checks on intervals if all registered database
 *          connections are still active. By using the simplest query form like fetching
 *          all tablenames on the connectionlevel, we trigger an error.
 *          Connection is checked every (check_interval) seconds. If an error is detected,
 *          check_interval is set to 1 seconds, reverting back to <check_interval> if the
 *          connection checks out again.
 */
class DbConnectionWatchDog : public QObject
{
    Q_OBJECT

public:
    /*!
     * \brief The constructor
     * \param check_interval in seconds.
     * \param parent    - The object that owns this connection guard.
     */
    explicit DbConnectionWatchDog( int check_interval = 5, QObject *_parent = 0 );

    ~DbConnectionWatchDog() {}

    /// Deleted Copy Constructor
    DbConnectionWatchDog( const DbConnectionWatchDog& source ) = delete;
    /// Deleted Assignment Operator
    DbConnectionWatchDog& operator=( const DbConnectionWatchDog& ) = delete;
    /// Deleted Move Constructor
    DbConnectionWatchDog( DbConnectionWatchDog&& ) = delete;
    /// Deleted Move Operator
    DbConnectionWatchDog& operator=( DbConnectionWatchDog&& ) = delete;

    void    start();

private:
    QPointer<QTimer>    m_pTimer;               ///< The timer object this guard is based on.
    int                 m_check_interval;       ///< The current check_interval the connections are checked.
    int                 m_original_interval;    ///< The check_interval the connections are checked if present.
    bool                m_dbc_valid;            ///< Private member reflecting the state. True for no problems, false for problems.

signals:
    void    signalDbConnected( const QString& db_name, bool is_open );

private slots:
    /*!
     * \brief Slot being called if the timer reaches its timeout value, starting the connectioncheck.
     */
    void    slotStartConnectionCheck();
};

}       /* End namespace components */
}       /* End namespace osdev      */

#endif  /* OSDEV_COMPONENTS_DBCONNECTIONWATCHDOG_H */
