/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ***************************************************************************/

#include "dbconnectionwatchdog.h"
#include "log.h"

#include <QSqlDatabase>
#include <QSqlError>

using namespace osdev::components;

DbConnectionWatchDog::DbConnectionWatchDog( int check_interval, QObject *_parent )
    : QObject( _parent )
    , m_pTimer( new QTimer( this ) )
    , m_check_interval( check_interval )
    , m_original_interval( check_interval )
    , m_dbc_valid( true )
{

    #if( QT_VERSION >= QT_VERSION_CHECK( 5, 7, 0) )
    {
        QObject::connect( m_pTimer.data(), &QTimer::timeout, this, &DbConnectionWatchDog::slotStartConnectionCheck );
    }
    #else
    {
        QObject::connect( m_pTimer.data(), SIGNAL( timeout() ), this, SLOT( slotStartConnectionCheck() ) );
    }
    #endif

}

void DbConnectionWatchDog::slotStartConnectionCheck()
{
    LogDebug( "[DbConnectionWatchDog::slotStartConnectionCheck]", QString( "Checking Database Connection...." ) );
    // Get all the connections from QSqlDatabase
    QStringList connection_names = QSqlDatabase::connectionNames();
    for( QString &conn_name : connection_names )
    {
        QSqlDatabase dbConn = QSqlDatabase::database( conn_name, false );


        // Check to see if we can execute a query. Not all databases drivers report a database as "open"
        // without actually accessing. It all depends on the Operating System and the connectiontype.
        if( 0 == dbConn.tables().count() )
        {
            // Check the error cause we didn't got any tablenames.
            switch( dbConn.lastError().type() )
            {
                case QSqlError::NoError:
                    LogWarning( "[DbConnectionWatchDog::slotStartConnectionCheck]", QString( "No Error or database unavailable on connection %1" ).arg( conn_name ) );
                    dbConn.open();
                    // Set the check interval to 1 second until the connection is back on.
                    m_check_interval = 1;
                    break;

                case QSqlError::ConnectionError:
                    LogWarning( "[DbConnectionWatchDog::slotStartConnectionCheck]", QString( "Connection Error on connection %1" ).arg( conn_name ) );
                    dbConn.open();
                    // Set the check interval to 1 second until the connection is back on.
                    m_check_interval = 1;
                    break;
                case QSqlError::UnknownError:
                    LogWarning( "[DbConnectionWatchDog::slotStartConnectionCheck]", QString( "Unknown Error on connection %1" ).arg( conn_name ) );
                    // Set the check interval to 1 second until the connection is back on.
                    m_check_interval = 1;
                    break;

                // The following Errors are not connection related but need to be here to satisfy the compiler.
                case QSqlError::StatementError:
                case QSqlError::TransactionError:
                    break;
                default:
                    LogInfo( "[DbConnectionWatchDog::slotStartConnectionCheck]", QString( "Database connection Healthy...." ) )
            }

            if( m_dbc_valid )
            {
                m_dbc_valid = false;
            }

            emit signalDbConnected( conn_name, false );
        }
        else
        {
            m_check_interval = m_original_interval;
            if( !m_dbc_valid )
            {
                LogInfo( "[DbConnectionWatchDog::slotStartConnectionCheck]", "Database connection restored.." );
                m_dbc_valid = true;
                emit signalDbConnected( conn_name, m_dbc_valid );
            }
        }
    }

    // Restart the watchDog
    this->start();
}

void DbConnectionWatchDog::start()
{
    m_pTimer->setInterval( m_check_interval * 1000 );
    m_pTimer->setSingleShot( true );

    m_pTimer->start();
}
