/*****************************************************************************
 * Copyright (c)2023 Peter M. Groen
 * This file is licensed under the MIT license found in the LICENSE file
 * in the root directory of this source tree.
 ******************************************************************************/

#include <gmock/gmock.h>
#include <gtest/gtest.h>

#include "Variant.h"

using namespace osdev::components;

TEST(DataTypesTest, BooleanTest)
{
    Variant oVariant(true);
    EXPECT_EQ(oVariant.getType(), Variant::Bool);
    EXPECT_TRUE(oVariant.CanConvert(Variant::Bool));
    EXPECT_TRUE(oVariant.toBool());
}

TEST(DataTypesTest, PositiveIntegerTest)
{
    int value = 42;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::Int);
    EXPECT_TRUE(oVariant.CanConvert(Variant::Int));
    EXPECT_EQ(oVariant.toInt(), 42);
}

TEST(DataTypesTest, NegativeIntegerTest)
{
    int value = -42;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::Int);
    EXPECT_TRUE(oVariant.CanConvert(Variant::Int));
    EXPECT_EQ(oVariant.toInt(), -42);
}

TEST(DataTypesTest, PositiveDoubleTest)
{
    double value = 42;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::Double);
    EXPECT_TRUE(oVariant.CanConvert(Variant::Double));
    EXPECT_EQ(oVariant.toDouble(), 42);
}

TEST(DataTypesTest, NegativeDoubleTest)
{
    double value = -42;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::Double);
    EXPECT_TRUE(oVariant.CanConvert(Variant::Double));
    EXPECT_EQ(oVariant.toDouble(), -42);
}

TEST(DataTypesTest, PositiveFloatTest)
{
    float value = 42.00;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::Float);
    EXPECT_TRUE(oVariant.CanConvert(Variant::Float));
    EXPECT_EQ(oVariant.toFloat(), 42.00);
}

TEST(DataTypesTest, NegativeFloatTest)
{
    float value = -42.00;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::Float);
    EXPECT_TRUE(oVariant.CanConvert(Variant::Float));
    EXPECT_EQ(oVariant.toFloat(), -42.00);
}

TEST(DataTypesTest, CharTest)
{
    char value = 'c';
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::Char);
    EXPECT_TRUE(oVariant.CanConvert(Variant::Char));
    EXPECT_EQ(oVariant.toChar(), 'c');
}

TEST(DataTypesTest, StringTest)
{
    std::string value = "This is a string";
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::String);
    EXPECT_TRUE(oVariant.CanConvert(Variant::String));
    EXPECT_EQ(oVariant.toString(), "This is a string");
}

TEST(DataTypesTest, UInt8Test)
{
    uint8_t value = 0xAA;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::UInt8);
    EXPECT_TRUE(oVariant.CanConvert(Variant::UInt8));
    EXPECT_EQ(oVariant.toUInt8(), 0xAA);
}

TEST(DataTypesTest, UInt16Test)
{
    uint16_t value = 0xAAAA;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::UInt16);
    EXPECT_TRUE(oVariant.CanConvert(Variant::UInt16));
    EXPECT_EQ(oVariant.toUInt16(), 0xAAAA);
}

TEST(DataTypesTest, UInt64Test)
{
    uint64_t value = 0xAAAAAAAAAAAAAAAA;
    Variant oVariant(value);
    EXPECT_EQ(oVariant.getType(), Variant::UInt64);
    EXPECT_TRUE(oVariant.CanConvert(Variant::UInt64));
    EXPECT_EQ(oVariant.toUInt64(), 0xAAAAAAAAAAAAAAAA);
}

TEST(DataTypesTest, CanConvertTest)
{
    uint64_t value = 0xAAAAAAAAAAAAAAAA;
    Variant oVariant(value);
    EXPECT_FALSE(oVariant.CanConvert(Variant::Bool));
    EXPECT_FALSE(oVariant.CanConvert(Variant::Int));
    EXPECT_FALSE(oVariant.CanConvert(Variant::Double));
    EXPECT_FALSE(oVariant.CanConvert(Variant::Float));
    EXPECT_FALSE(oVariant.CanConvert(Variant::Char));
    EXPECT_FALSE(oVariant.CanConvert(Variant::String));
    EXPECT_FALSE(oVariant.CanConvert(Variant::UInt8));
    EXPECT_FALSE(oVariant.CanConvert(Variant::UInt16));
    EXPECT_TRUE(oVariant.CanConvert(Variant::UInt64));
}

TEST(DataTypesTest, GetWrongTypeTest)
{
    uint64_t value = 0xAAAAAAAAAAAAAAAA;
    Variant oVariant(value);
    EXPECT_ANY_THROW(oVariant.toBool());
    EXPECT_ANY_THROW(oVariant.toInt());
    EXPECT_ANY_THROW(oVariant.toDouble());
    EXPECT_ANY_THROW(oVariant.toFloat());
    EXPECT_ANY_THROW(oVariant.toChar());
    EXPECT_ANY_THROW(oVariant.toString());
    EXPECT_ANY_THROW(oVariant.toUInt8());
    EXPECT_ANY_THROW(oVariant.toUInt16());
    EXPECT_NO_THROW(oVariant.toUInt64());
}

