/* ****************************************************************************
 * Copyright 2019 Open Systems Development BV                                 *
 *                                                                            *
 * Permission is hereby granted, free of charge, to any person obtaining a    *
 * copy of this software and associated documentation files (the "Software"), *
 * to deal in the Software without restriction, including without limitation  *
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,   *
 * and/or sell copies of the Software, and to permit persons to whom the      *
 * Software is furnished to do so, subject to the following conditions:       *
 *                                                                            *
 * The above copyright notice and this permission notice shall be included in *
 * all copies or substantial portions of the Software.                        *
 *                                                                            *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,   *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL    *
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING    *
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER        *
 * DEALINGS IN THE SOFTWARE.                                                  *
 * ****************************************************************************/
#pragma once

#include <sys/syslog.h>
#include <syslog.h>
#include <cstdio>
#include <string>

namespace osdev::components::daemon {

class daemonlog
{
public:
    /**
     *  Initialize the logger
     *  @param  daemon_name
     */
    static void init(const std::string &daemon_name)
    {
        // m_daemon_name = daemon_name;
        openlog(daemon_name.c_str(), LOG_PID, LOG_DAEMON);
    }

    /**
     *  main logger with priority LOG_X
     *  @param  message
     *  @param  priority
     */
    static void log(const std::string &message, std::int32_t priority)
    {
        syslog(priority, "%s", message.c_str());
    }

    /**
     *  debug-level messages
     *  @param  message
     */
    static void debug(const std::string &message)
    {
        log(message, LOG_DEBUG);
    }

    /**
     *  informational
     *  @param  message
     */
    static void info(const std::string &message)
    {
        log(message, LOG_INFO);
    }

    /**
     *  normal but significant condition
     *  @param  message
     */
    static void notice(const std::string &message)
    {
        log(message, LOG_NOTICE);
    }

    /**
     *  Warning conditions
     *  @param  message
     */
    static void warning(const std::string &message)
    {
        log(message, LOG_WARNING);
    }

    /**
     *  error conditions
     *  @param  message
     */
    static void error(const std::string &message)
    {
        log(message, LOG_ERR);
    }

    /**
     *  critical conditions
     *  @param  message
     */
    static void critical(const std::string &message)
    {
        log(message, LOG_CRIT);
    }

    /**
     *  action must be taken immediately
     *  @param  message
     */
    static void alert(const std::string &message)
    {
        log(message, LOG_ALERT);
    }

    /**
     *  system is unusable
     *  @param  message
     */
    static void emergency(const std::string &message)
    {
        log(message, LOG_EMERG);
    }

    /**
     *  shutdown the logger
     */
    static void shutdown()
    {
        closelog();
    }

private:
    static std::string priority_str(std::int32_t priority)
    {
        switch(priority)
        {
            case LOG_EMERG: return "emergency";
            case LOG_ALERT: return "alert";
            case LOG_CRIT: return "critical";
            case LOG_ERR: return "error";
            case LOG_WARNING: return "warning";
            case LOG_NOTICE: return "notice";
            case LOG_INFO: return "info";
            case LOG_DEBUG: return "debug";
            default: return "unknown_priority";
        }
    }

    std::string m_daemon_name;

};

// std::string osdev::components::daemon::m_daemon_name{};

}