#include "crypter.h"

// std
#include <iomanip>
#include <memory>
#include <sstream>
#include <stdexcept>
#include <system_error>

// openssl
#include <openssl/evp.h>

// osdev::components
#include "log.h"
#include "scopeguard.h"

using namespace osdev::components;

namespace {

const EVP_MD* getAlgo( Crypter::AlgorithmEnum algo )
{
    switch ( algo )
    {
        case Crypter::AlgorithmEnum::MD0:
            return EVP_md_null();
        case Crypter::AlgorithmEnum::MD2:
            // return EVP_md2();
        case Crypter::AlgorithmEnum::MD5:
            return EVP_md5();
        case Crypter::AlgorithmEnum::MDC2:
            // deprecated.
            // md = EVP_mdc2();
            return EVP_md_null();
        case Crypter::AlgorithmEnum::SHA1:
            return EVP_sha1();
        case Crypter::AlgorithmEnum::SHA224:
            return EVP_sha224();
        case Crypter::AlgorithmEnum::SHA256:
            return EVP_sha256();
        case Crypter::AlgorithmEnum::SHA384:
            return EVP_sha384();
        case Crypter::AlgorithmEnum::SHA512:
            return EVP_sha512();
        case Crypter::AlgorithmEnum::RIPEMD160:
            return EVP_ripemd160();
    }

    throw std::invalid_argument("Crypto algorithm not found.");
}

} // anonymous

Crypter::Crypter()
{
}

std::string Crypter::encrypt( const std::string& message, Crypter::AlgorithmEnum algo )
{
    // Create the environment
    auto md = getAlgo(algo);

#if OPENSSL_VERSION_NUMBER >= 0x1010008fL
    auto mdContext = std::unique_ptr<EVP_MD_CTX, decltype(&EVP_MD_CTX_free)>(
                EVP_MD_CTX_create(),
                [](EVP_MD_CTX* ptr){EVP_MD_CTX_free(ptr);});

    // This will call EVP_cleanup if the guard goes out of scope.
    ScopeGuard oGuard( &EVP_PBE_cleanup );
#else
    auto mdContext = std::unique_ptr<EVP_MD_CTX, decltype(&EVP_MD_CTX_destroy)>(
                EVP_MD_CTX_create(),
                [](EVP_MD_CTX* ptr){EVP_MD_CTX_destroy(ptr);});

    // This will call EVP_cleanup if the guard goes out of scope.
    ScopeGuard oGuard( &EVP_cleanup );
#endif

    (void)oGuard; // Satisfy the compiler for unused variables.

    auto errorCode = EVP_DigestInit_ex( mdContext.get(), md, NULL );
    if( 1 != errorCode )
    {
        LogError( "[Crypter::encrypt]", std::string( "No encryption digest environment created." ) );
        throw std::system_error( errorCode, std::system_category(), "No encryption digest environment created." );
    }

    // Update the environment with the message
    errorCode = EVP_DigestUpdate( mdContext.get(), message.c_str(), message.length() );
    if( 1 != errorCode )
    {
        LogError( "[Crypter::encrypt]", std::string("Digest failed.") );
        throw std::system_error( errorCode, std::system_category(), "Digest failed.." );
    }

    // End the Digest so we can read the crypted message.
    unsigned int mdLen;
    unsigned char mdValue[EVP_MAX_MD_SIZE];
    errorCode = EVP_DigestFinal_ex( mdContext.get(), mdValue, &mdLen );
    if( 1 != errorCode )
    {
        LogError( "[Crypter::encrypt]", std::string("There was an error closing the digest environment.") );
        throw std::system_error( errorCode, std::system_category(), "There was an error closing the digest environment." );
    }

    // If we got here, all went well. We retrieve the crypted message
    // through a stringstream : convert to hex, padding and width and return the string.
    std::stringstream ss;
    for( unsigned int nIndex = 0; nIndex < mdLen; nIndex++ )
    {
        ss << std::hex << std::setw( 2 ) << std::setfill( '0' ) << static_cast<int>( mdValue[nIndex] );
    }

    return ss.str();
}
