#ifndef OSDEV_COMPONENTS_OBJECTDATA_H
#define OSDEV_COMPONENTS_OBJECTDATA_H

#include <QSharedPointer>
#include <QStringList>

#include "connectordata.h"

namespace osdev {
namespace components {

/*
 *  _____________________________________
 * < Excellent day to have a rotten day. >
 *  -------------------------------------
 *    \
 *     \
 *         .--.
 *        |o_o |
 *        |:_/ |
 *       //   \ \
 *      (|     | )
 *     /'\_   _/`\
 *     \___)=(___/
 *
 */
/*!
 * \brief The ObjectData class
 */

class ObjectData
{
public:
    /*!
     * \brief Default constructor;
     */
    ObjectData();

    /*!
     * \return The object type.
     */
    const QString& getObjectType() const { return m_objectType; }

    /*!
     * \return The object id.
     */
    const QString& getObjectId() const { return m_objectId; }

    /*!
     * \return The target name.
     */
    const QString& getTargetName() const { return m_targetName; }

    /*!
     * \return The target action.
     */
    const QString& getTargetAction() const { return m_targetAction; }

    /*!
     * \return The key field.
     */
    const QString& getKeyField() const { return m_keyField; }

    /**
     * @return The key value.
     */
    const QString& getKeyValue() const { return m_keyValue; }

    /**
     * @return The foreign key field.
     */
    const QString& getForeignKeyField() const { return m_foreignKeyField; }

    /**
     * @return The target field.
     */
    const QString& getTargetField() const { return m_targetField; }

    /**
     * @return The target field reset value.
     */
    const QString& getTargetResetValue() const { return m_targetResetValue; }

    /**
     * @return Flag value that indicates whether persistent or non persistent timestamps should be used.
     */
    bool getNonPersistentTimestampUsage() const { return m_useNonPersistentTimestamp; }

    /**
     * @return Size of the buffer that is used for storing the non persistent timestamps.
     */
    unsigned int getNonPersistentTimestampBufferSize() const { return m_npTimestampBufferSize; }

    /*!
     * \brief Sets the ObjectType.
     * \param objectType The type to set.
     */
    void setObjectType(const QString& objectType);

    /*!
     * \brief setObjectId Sets the object id.
     * \param objectId The id to set.
     */
    void setObjectId(const QString& objectId);

    /*!
     * \brief Sets the TargetName.
     * \param targetName The target name to set.
     */
    void setTargetName(const QString& targetName);

    /*!
     * \brief Sets the TargetAction.
     * \param targetAction The action to set.
     */
    void setTargetAction(const QString& targetAction);

    /*!
     * \brief Sets the KeyField.
     * \param keyField The key field to set.
     */
    void setKeyField(const QString& keyField);

    /**
     * @brief Sets the keyvalue for a match in a DataFilter
     * @param keyValue The value to set.
     */
    void setKeyValue(const QString& keyValue);

    /*!
     * \brief Sets the foreign key field (used for batch/merge update).
     * \param foreignKeyField The foreign key field to set.
     */
    void setForeignKeyField(const QString& foreignKeyField);

    /**
     * @brief Set the field on which a merge/batch update operation operates.
     * @param targetField The field to change values on in batch/merge updates.
     */
    void setTargetField(const QString& targetField);

    /**
     * @brief Set the reset value to which the targetfield is set in the decouple step of the merge update operation.
     * @param targetResetValue The reset value to use in a field reset.
     */
    void setTargetResetValue(const QString& targetResetValue);

    /**
     * @brief Set the flag value that indicates whether persistent or non persistent timestamps should be used.
     * @param value The value to set.
     */
    void setNonPersistentTimestampUsage(bool value);

    /**
     * @brief Set the size (number of timestamps) of the buffer used to store non persistent timestamps.
     * @param value The value to set.
     */
    void setNonPersistentTimestampBufferSize(unsigned int value);

    /*!
     * \return The input names.
     */
    const QStringList getInputNames() const;

    /*!
     * \brief Get the id of the input element identified by name.
     * \param inputName The name that identifies the input element.
     * \return input id or a null string when the input does not have an id.
     */
    const QString& getInputId(const QString& inputName) const;

    /*!
     * \brief Gets the InputType for the specified input name.
     * \param inputName The name for which to get the input type.
     * \return The InputType for the specified input name.
     */
    const QString& getInputType(const QString& inputName) const;

    /*!
     * \brief Gets the InputDefault for the specified input name.
     * \param inputName The inout name for chich to get the default.
     * \return The InputDefault for the specified input name.
     */
    const QString& getInputDefault(const QString& inputName) const;

    /*!
     * \brief Get the excludeFromIdentityCheck flag for the specified input name.
     * \param inputName The name for which to get the excludeFromIdentityCheck flag.
     * \return The excludeFromIdentityCheck flag.
     */
    bool getInputExcludeFromIdentityCheck(const QString& inputName) const;

    /*!
     * \return The output names.
     */
    const QStringList getOutputNames() const;

    /*!
     * \brief Gets the OutputType for the specified output name.
     * \param outputName The output name for which to get the type.
     * \return The OutputType for the specified output name.
     */
    const QString& getOutputType(const QString& outputName) const;

    /*!
     * \brief Gets the OutputDefault for the specified output name.
     * \param outputName The name for which to get the default.
     * \return The OutputDefault for the specified output name.
     */
    const QString& getOutputDefault(const QString& outputName) const;

    /*!
     * \brief Sets the InputData.
     * \param name The name of the input data.
     * \param type The type of the input data.
     * \param id The id of the input data. If there is no id a null string must be used.
     * \param defaultValue The defaultValue of the input data.
     * \param excludeFromIdentityCheck Flag that indicates whether this data should participate in the record identity check.
     */
    void setInputData(const QString& name, const QString& type, const QString& id, const QString& defaultValue, bool excludeFromIdentityCheck);

    /*!
     * \brief setOutputData
     * \param name The name of the output data.
     * \param type The type of the output data.
     * \param defaultValue The defaultValue of the output data.
     */
    void setOutputData(const QString& name, const QString& type, const QString& defaultValue);

private:
    QString m_objectType;                 ///<    Type of Object
    QString m_objectId;                   ///<    UUID of the object
    QString m_targetName;                 ///<    TargetTable this object represent
    QString m_targetAction;               ///<    What action should taken on the target
    QString m_keyField;                   ///<    What TargetField is used for Querying
    QString m_keyValue;                   ///<    The value we're querying for.
    QString m_foreignKeyField;            ///<    Contains reference to a foreign table.
    QString m_targetField;                ///<    The field on which batch/merge operations operate.
    QString m_targetResetValue;           ///<    The value to which fields are reset when they are not part of the merge set.
    bool m_useNonPersistentTimestamp;     ///<    Use persistent or non persistent timestamp. Default is false.
    unsigned int m_npTimestampBufferSize; ///<    Number of timestamps that can be stored (circular buffer). Default is 10.

    QHash<QString, QSharedPointer<ConnectorData>> m_qhInputs;  ///<    Hash of all InputConnectors by Name
    QHash<QString, QSharedPointer<ConnectorData>> m_qhOutputs; ///<    Hash of all OutputConnectors by Name
};

} // namespace components
} // namespace osdev

#endif // OSDEV_COMPONENTS_OBJECTDATA_H
