#ifndef OSDEV_COMPONENTS_JOBDATA_H
#define OSDEV_COMPONENTS_JOBDATA_H

#include <QHash>
#include <QVariant>
#include <QString>
#include <QDate>
#include <QTime>
#include <QStringList>

namespace osdev {
namespace components {

/*
 *  _________________________________________
 * / Despite the best efforts of a quantum   \
 * | bigfoot drive (yes I know everyone told |
 * | me they suck, now I know they were      |
 * | right) 2.1.109ac1 is now available      |
 * |                                         |
 * \ -- Alan Cox announcing Linux 2.1.109ac1 /
 *  -----------------------------------------
 *   \
 *    \
 *        .--.
 *       |o_o |
 *       |:_/ |
 *      //   \ \
 *     (|     | )
 *    /'\_   _/`\
 *    \___)=(___/
 */
/*!
 * \brief   A simple class to hold all info needed to schedule a job.
 */

class JobData
{
public:
    /// Default constructor
    JobData();

    /*!
     * \brief   Construct the JobData object and set all the parameters
     * \param   _jobName    - Name of the job (Coming from the config)
     * \param   _paramList  - The parameters as coming from the config
     * \param   _runTime    - Time this job has to run. (Coming from the configuration.
     * \param   _runDate    - (Local)Date this job has to run.
     * \param   _interval   - The interval this job has to be rescheduled
     * \param   _targetObject   - The caelus object this job is intended for.
     */
    JobData( const QString& _jobName, const QHash<QString, QVariant>& _paramList,
             const QTime& _runTime, const int _interval, const QString& _targetObject,
             const QDate& _runDate = QDate::currentDate() );

    /* Copy Constructor */
    JobData( const JobData& source );

    /* Assignment operator */
    JobData& operator=( const JobData& origin );

    /** D'tor */
    virtual ~JobData();

    /*!
     * \return  The name of the job used for registration.
     *          It will be set by "setJobName( const QString& _jobName )"
     */
    const QString& jobName() const { return m_jobName; }
    void setJobName( const QString& _jobName ) { m_jobName = _jobName; }

    /*!
     * \return   Get the targetobject this job is intended for.
     */
    const QString& targetObject() const { return m_targetObject; }
    void setTargetObject( const QString& _targetObject ) { m_targetObject = _targetObject; }

    /*!
     * \return  The date this job should run. If this returns empty, it should run each and every day.
     *          It will be set by "setRunDate( const QDate& _runDate )".This is LocalTime....
     */
    const QDate& runDate() const { return m_runDate; }
    void setRunDate( const QDate& _runDate ) { m_runDate = _runDate; }

    /*!
     * \return  The time this job should run. It cannot be empty.
     *          It will be set by "setRunTime( const QTime& _runTime )"
     */
    const QTime& runTime() const { return m_runTime; }
    void setRunTime( const QTime& _runTime ) { m_runTime = _runTime; }

    /*!
     * \brief   The interval the job will be re-scheduled to.
     * \return
     */
    int runInterval() const { return m_interval; }
    void setRunInterval( const int _interval ) { m_interval = _interval; }

    // ========================================================================
    // == Implemented in the cpp file
    // == ---------------------------------------------------------------------
    /*!
     * \brief   Adds a parameter to the parameter list this job uses.
     */
    void addParameter( const QString& _varName, const QVariant &_value );

    /*!
     * \brief   Replace the set of parameters with the new one. The old list will be erased.
     */
    void setParameters( const QHash < QString, QVariant > &_paramList );

    /*!
     * \return  Get the value, belonging to the variable name from the hash.
     *          Empty if the parameter name doesn't exist.
     */
    QVariant paramValue( const QString& _varName ) const;

    /*!
     * \return  Get all parameter names from the list. Empty if none exist.
     */
    QStringList varNames() const;

    /*!
     * \return  The entire parameter list
     */
     QHash<QString, QVariant> paramList() const { return m_parameters; }

private:
    /*!
     *  \brief  Copies this instance into the specified destination.
     *  \param[in,out] dest Pointer to the destination to which the properties of this instance are copied.
     */
    void copyImpl( JobData* dest ) const;

    QString                     m_jobName;      ///< The name of the job used for registration
    QDate                       m_runDate;      ///< The date this job should run. Empty if daily.
    QTime                       m_runTime;      ///< The time this job should run.
    int                         m_interval;     ///< The interval the jod should be scheduled on.
    QString                     m_targetObject; ///< The plugin or object this job is intended for. Used as input to the pluginmanager.
    QHash<QString, QVariant>    m_parameters;   ///< Parameters used by the job.
};

}   /* End namespace components */
}   /* End namespace osdev */

#endif  /* OSDEV_COMPONENTS_JOBDATA_H */
