#ifndef OSDEV_COMPONENTS_DCXMLMODELMAPPING_H
#define OSDEV_COMPONENTS_DCXMLMODELMAPPING_H

#include "dcxmletlnetwork.h"
#include "dcxmletlservices.h"

#include <QHash>
#include <QObject>
#include <QSharedPointer>
#include <QStringList>
#include <memory>

namespace osdev {
namespace components {

/*! \brief  This component locates any configuration information needed
 *          to initialize the application and its components.
 *          Parses this information and make it available to other
 *          components that need it.
 *
 *          This component is implemented as a singleton. XML has the
 *          "habit" of expand approx. 15 times the file size so with this
 *          in mind, it is more future proof to load this only once into memory.
 */
/*
 *    ________________________________________
 *   / History, n.:                           \
 *   |                                        |
 *   | Papa Hegel he say that all we learn    |
 *   | from history is that we                |
 *   | learn nothing from history. I know     |
 *   | people who can't even learn from       |
 *   | what happened this morning. Hegel must |
 *   | have been taking the long view.        |
 *   |                                        |
 *   | -- Chad C. Mulligan, "The Hipcrime     |
 *   \ Vocab"                                 /
 *    ----------------------------------------
 *      \
 *       \
 *           .--.
 *          |o_o |
 *          |:_/ |
 *         //   \ \
 *        (|     | )
 *       /'\_   _/`\
 *       \___)=(___/
 */

using network_container = QHash<QString, QSharedPointer<DcXmlEtlNetwork>>;

class DcXmlModelMapping
{
public:
    /// @return the one and only instance of the config object.
    static DcXmlModelMapping& Instance();

    // =========================================================
    /// The constructor
    DcXmlModelMapping();

    /// The destructor
    virtual ~DcXmlModelMapping();

    /// Deleted copy-constructor
    DcXmlModelMapping(const DcXmlModelMapping&) = delete;
    /// Deleted assignment constructor
    DcXmlModelMapping& operator=(const DcXmlModelMapping&) = delete;
    /// Deleted move-constructor
    DcXmlModelMapping(DcXmlModelMapping&&) = delete;
    /// Deleted move-operator
    DcXmlModelMapping& operator=(DcXmlModelMapping&&) = delete;

    // ==========================================================
    /// @brief  Loads the configuration from the given filename
    /// @param  fileName - The configuration file in XML format.
    bool loadConfiguration(const QString& configDir, const QString& fileName);

    bool loadConfiguration(const QString& fileName);

    /*! @brief  Retrieve a list of services configured in the ModelMapper configuration
     *  @return A StringList of all serviceNames.
     */
    QStringList getServices() const;
    QString getServiceId(const QString& _serviceName) const;

    QStringList getNetworksOfService(const QString& _serviceName) const;
    QList<QSharedPointer<ObjectData>> getObjectsOfNetwork(const QString& _networkName) const;
    Connections getConnectionsOfNetwork(const QString& _networkName) const;

    QStringList getInputVariables(const QString& _networkName) const;
    QStringList getOutputVariables(const QString& _networkName) const;

    QThread::Priority getPrioByNetworkName(const QString& _networkName) const;

private:
    void cleanInternals();                                                               ///< Cleanup the internal structures and destroy all objects.
    QSharedPointer<DcXmlEtlNetwork> findNetworkByName(const QString& networkName) const; ///< Find the object that contains a config with this name.

    static std::unique_ptr<DcXmlModelMapping> s_instance; ///< Instantiated ModelMapper Configuration object.

    // Internal pointers......
    QSharedPointer<DcXmlEtlServices> m_services; ///< Instantiated Services Configuration object.
    network_container m_networks;                ///< Collection of instantiated network configuration objects.
};

} // namespace components
} // namespace osdev

#endif // OSDEV_COMPONENTS_DCXMLMODELMAPPING_H
