#include "dcxmlmodelmapping.h"
#include "dcxmlconfig.h"

using namespace osdev::components;

// The only instance of the singleton Modelmapper Config Parser.
std::unique_ptr<DcXmlModelMapping> DcXmlModelMapping::s_instance(nullptr);

DcXmlModelMapping& DcXmlModelMapping::Instance()
{
    if (nullptr == s_instance) {
        s_instance = std::unique_ptr<DcXmlModelMapping>(new DcXmlModelMapping());
    }

    return *s_instance;
}

DcXmlModelMapping::DcXmlModelMapping()
    : m_services(nullptr)
    , m_networks()
{
}

DcXmlModelMapping::~DcXmlModelMapping()
{
    this->cleanInternals();
}

bool DcXmlModelMapping::loadConfiguration(const QString& configDir, const QString& fileName)
{
    // Check if the internal service parser is instantiated. If not, do so.
    // The network parsers will be parsed "on the way".
    if (nullptr != m_services || !m_networks.isEmpty()) {
        // clean up the internal structures.. We obviously want to (re)load the configuration.
        cleanInternals();
    }

    // Start the services configuration...
    m_services = QSharedPointer<DcXmlEtlServices>(new DcXmlEtlServices(configDir + "/" + fileName));
    for (const QString& serviceName : m_services->getServiceNames()) {
        QSharedPointer<DcXmlEtlNetwork> pNetwork = QSharedPointer<DcXmlEtlNetwork>(new DcXmlEtlNetwork(configDir + "/" + m_services->getServiceConfigByName(serviceName)));

        if (nullptr != pNetwork) {
            m_networks.insert(serviceName, pNetwork);
        }
    }

    return true;
}

bool DcXmlModelMapping::loadConfiguration(const QString& fileName)
{
    return loadConfiguration(DCXmlConfig::Instance().getConfigPath(), fileName);
}

QStringList DcXmlModelMapping::getServices() const
{
    return QStringList(m_networks.keys());
}

QString DcXmlModelMapping::getServiceId(const QString& _serviceName) const
{
    return m_services->getServiceIdByName(_serviceName);
}

QStringList DcXmlModelMapping::getNetworksOfService(const QString& _serviceName) const
{
    return m_networks.value(_serviceName)->getNetworkNames();
}

QList<QSharedPointer<ObjectData>> DcXmlModelMapping::getObjectsOfNetwork(const QString& _networkName) const
{
    // Find the network pointer containing the networkName.
    QSharedPointer<DcXmlEtlNetwork> pNetwork = findNetworkByName(_networkName);
    if (nullptr != pNetwork) {
        return pNetwork->getObjectsOfNetwork(_networkName);
    }
    return QList<QSharedPointer<ObjectData>>();
}

Connections DcXmlModelMapping::getConnectionsOfNetwork(const QString& _networkName) const
{
    // Find the network pointer containing the networkName.
    QSharedPointer<DcXmlEtlNetwork> pNetwork = findNetworkByName(_networkName);
    if (nullptr != pNetwork) {
        return pNetwork->getConnectionsOfNetwork(_networkName);
    }
    return Connections();
}

QStringList DcXmlModelMapping::getInputVariables(const QString& _networkName) const
{
    // Find the network pointer containing the networkName.
    QSharedPointer<DcXmlEtlNetwork> pNetwork = findNetworkByName(_networkName);
    if (nullptr != pNetwork) {
        return pNetwork->getInputVariables(_networkName);
    }
    return QStringList();
}

QStringList DcXmlModelMapping::getOutputVariables(const QString& _networkName) const
{
    QSharedPointer<DcXmlEtlNetwork> pNetwork = findNetworkByName(_networkName);
    if (nullptr != pNetwork) {
        return pNetwork->getOutputVariables(_networkName);
    }
    return QStringList();
}

QThread::Priority DcXmlModelMapping::getPrioByNetworkName(const QString& _networkName) const
{
    // Find the network pointer containing the networkName.
    QSharedPointer<DcXmlEtlNetwork> pNetwork = findNetworkByName(_networkName);
    if (nullptr != pNetwork) {
        return pNetwork->getPrioByNetworkName(_networkName);
    }
    return QThread::NormalPriority;
}

void DcXmlModelMapping::cleanInternals()
{
    // Retrieve every network object and destroy it...
    if (!m_networks.isEmpty()) {
        for (const QString& serviceName : QStringList(m_networks.keys())) {
            // Clear the reference. If it was the last (only) reference
            // the pointer wil be deleted.
            m_networks.take(serviceName).clear();
        }
    }

    m_services.clear();
}

QSharedPointer<DcXmlEtlNetwork> DcXmlModelMapping::findNetworkByName(const QString& networkName) const
{
    for (QSharedPointer<DcXmlEtlNetwork> pNetwork : m_networks.values()) {
        if (pNetwork->getNetworkNames().contains(networkName)) {
            return pNetwork;
        }
    }
    return QSharedPointer<DcXmlEtlNetwork>();
}
